#include <iostream>
#include <fstream>
#include <boost/algorithm/string.hpp>
#include <vector>
#include <map>
#include <sys/stat.h>
#include <thread>
#include <pwd.h>
#include <mutex>



#include "functions.hpp"
#include "external.hpp"
#include "evidence.hpp"
#include "statistics.hpp"
#include "ThreadPool.h"
#include "checkpl.hpp"
#include "checkad.hpp"
#include "filter.hpp"
#include "fasta.hpp"
#include "haploview.hpp"
#include "genepop.hpp"
#include "arlequin.hpp"
#include "hard_filter.hpp"


using namespace std;

string Program_name = "vcfx";
string Program_version = "2.0.6b";
string Program_author = "Erick C. Castelli and Celso T. Mendes-Junior";
string Program_date = "May 2021";
string v_website = "www.castelli-lab.net/apps/vcfx";
string Program_contact = "erick.castelli@unesp.br";
int screen_size = 80;

string v_input = "";
string v_output = "";

int min_cov = 12;
float min_proportion = 0.3f;
float max_proportion = 0.7f;
float max_missing_per_variant = 0.10;
int buffer = 500;
string vstart = "0";
string vend = "0";
string v_chr = "";
int v_quiet = 0;

mutex mtx;
string v_vcf;
string v_out;
int count_total = 0;
int count_rejected = 0;
int count_monomorfic = 0;
int count_pass = 0;
int count_warn = 0;
int count_not_all = 0;
int count_mono_alt = 0;
int count_singleton = 0;
int count_missing = 0;

string tag = "PASS,PASS_Evidence,PASS_Reference,PASS_Reference_Evidence,MONO_ALT";

vector <string> warnings;

float v_gl = 0.99;
int alpha = 8;
float delta = 0.20f;


int v_concurentThreadsSupported = std::thread::hardware_concurrency() / 2;
int v_threads = v_concurentThreadsSupported;


auto clock_start = std::chrono::steady_clock::now();

string v_reference = "";
string v_bed = "";
string v_pop;
int keep_info = 0;


float v_qd = 2;
float v_fs = 60;
float v_sor = 3;
float v_mq = 40;
float v_mqr_min = -2.5;
float v_mqr_max = 2.5;
float v_rprs_min = -1;
float v_rprs_max = 3.5;



// ************ MAIN **************

void main_help(void)
{
	screen_message (screen_size, 0, "", 1, 0);
	screen_message (screen_size, 0, Program_name, 1, 0);
	screen_message (screen_size, 0, "", 1, 0);
	screen_message (screen_size, 2, "* Author  : " + Program_author, 1, 0);
	screen_message (screen_size, 2, "* Contact : " + Program_contact, 1, 0);
	screen_message (screen_size, 2, "* Version : " + Program_version, 1, 0);
	screen_message (screen_size, 2, "* Website : " + v_website, 1, 0);
	screen_message (screen_size, 0, "", 1, 0);
	screen_message (screen_size, 2, "Checking Tools", 1, 0);
	screen_message (screen_size, 5, "checkpl     introduce missing alleles on genotypes with low likelihood", 1, 0);
	screen_message (screen_size, 5, "checkad     introduce missing alleles on unbalanced genotypes", 1, 0);
	screen_message (screen_size, 5, "statistics  print some VCF statistics", 1, 0);
	screen_message (screen_size, 5, "hfilter     hard-filtering variants", 1, 0);
	screen_message (screen_size, 5, "evidence    annotate variants with some quality control parameters", 1, 0);
	screen_message (screen_size, 5, "filter      keep only variants annotated with specific filter tags", 1, 0);
	screen_message (screen_size, 0, "", 1, 0);

	screen_message (screen_size, 2, "Convertion Tools", 1, 0);
	screen_message (screen_size, 5, "fasta       create complete sequences from phased VCF", 1, 0);
	screen_message (screen_size, 5, "transcript  craate complete sequences from phased VCF using a BED file", 1, 0);
	screen_message (screen_size, 5, "haploview   recode VCF to PED for haploview (unphased)", 1, 0);
	screen_message (screen_size, 5, "arlequin    recode VCF to ARP for Arlequin", 1, 0);
	screen_message (screen_size, 5, "genepop     recode VCF to GenePop format", 1, 0);
	screen_message (screen_size, 0, "", 1, 0);
	return;
}




int main(int argc, const char * argv[]) {
	
	
	
    if (argc == 1)
    {
		main_help();
        return (0);
    }
	
	
    int a;
    for (a = 2; a < argc; a++)
    {
		
        string str = argv[a];
		
        if (str.find("input=") != string::npos)
        {
            v_input = str.substr(6);
			continue;
        }
		
		else if (str.find("output=") != string::npos)
        {
            v_output = str.substr(7);
			continue;
        }

		else if (str.find("start=") != string::npos)
		{
			vstart = str.substr(6);
			continue;
		}
		
		else if (str.find("end=") != string::npos)
		{
			vend = str.substr(4);
			continue;
		}
		
		else if (str.find("tag=") != string::npos)
		{
			tag = str.substr(4);
			if (tag == "") {tag = "PASS";}
			continue;
		}

		else if (str.find("mincov=") != string::npos)
		{
			try {
				min_cov = atoi(str.substr(7).c_str());
			}
			catch (const std::exception& e) {
				min_cov = 15;
			}
			continue;
		}
		
		else if (str.find("minprop=") != string::npos)
		{
			try {
				min_proportion = atof(str.substr(8).c_str());
			}
			catch (const std::exception& e) {
				min_proportion = 0.35;
			}
			continue;
		}


		else if (str.find("maxprop=") != string::npos)
		{
			try {
				max_proportion = atof(str.substr(8).c_str());
			}
			catch (const std::exception& e) {
				max_proportion = 0.65;
			}
			continue;
		}
		
		else if (str.find("maxmiss=") != string::npos)
		{
			try {
				max_missing_per_variant = atof(str.substr(8).c_str());
			}
			catch (const std::exception& e) {
				max_missing_per_variant = 0.10;
			}
			continue;
		}
		
		

		else if (str.find("buffer=") != string::npos)
		{
			try {
				buffer = stoi(str.substr(7).c_str());
			}
			catch (const std::exception& e) {
				buffer = 500;
			}
			continue;
		}
		
		
		else if (str.find("threads=") != string::npos)
		{
			try {
				v_threads = stoi(str.substr(8).c_str());
			}
			catch (const std::exception& e) {
				v_threads = v_concurentThreadsSupported;
			}
			continue;
		}
		
		
		else if (str.find("gl=") != string::npos)
		{
			try {
				v_gl = atof(str.substr(3).c_str());
			}
			catch (const std::exception& e) {
				v_gl = 0.99;
			}
			continue;
		}
		
		
		
		else if (str.find("chr=") != string::npos)
		{
			v_chr = str.substr(4);
			continue;
		}
		
		
		else if (str.find("delta=") != string::npos)
		{
			try {
				delta = atof(str.substr(6).c_str());
				if (delta > 0.40f)
				{
					delta = 0.40f;
					warnings.push_back("Delta was reduced to the maximum allowed: 0.40");
				}
			}
			catch (const std::exception& e) {
				delta = 0.2f;
			}
			continue;
		}

		
		else if (str.find("alpha=") != string::npos)
		{
			try {
				alpha = stoi(str.substr(6).c_str());
			}
			catch (const std::exception& e) {
				alpha = 8;
			}
			continue;
		}
		
		else if (str.find("reference=") != string::npos)
		{
			v_reference = str.substr(10);
			continue;
		}
		
		else if (str.find("bed=") != string::npos)
		{
			v_bed = str.substr(4);
			continue;
		}

		else if (str.find("pop=") != string::npos)
		{
			v_pop= str.substr(4);
			continue;
		}

		else if (str.find("QD=") != string::npos)
		{
			v_qd= stof(str.substr(3));
			continue;
		}

		else if (str.find("FS=") != string::npos)
		{
			v_fs= stof(str.substr(3));
			continue;
		}
		
		else if (str.find("SOR=") != string::npos)
		{
			v_sor= stof(str.substr(4));
			continue;
		}
		
		else if (str.find("MQ=") != string::npos)
		{
			v_mq= stof(str.substr(3));
			continue;
		}
		
		else if (str.find("RPRS_min=") != string::npos)
		{
			v_rprs_min= stof(str.substr(9));
			continue;
		}

		else if (str.find("MQR_min=") != string::npos)
		{
			v_mqr_min= stof(str.substr(8));
			continue;
		}

		else if (str.find("MQR_max=") != string::npos)
		{
			v_mqr_max= stof(str.substr(8));
			continue;
		}
		
		else if (str.find("RPRS_max=") != string::npos)
		{
			v_rprs_max= stof(str.substr(9));
			continue;
		}
		
		else if (str.find("--quiet") != string::npos)
		{
			v_quiet= 1;
			continue;
		}
		
		else if (str.find("--info") != string::npos)
		{
			keep_info= 1;
			continue;
		}

    }
	
	
	
    if (strcmp(argv[1],"evidence") == 0)
	{
		main_evidence();
	}

	
	else if (strcmp(argv[1],"statistics") == 0)
	{
		main_statistics();
	}

	
	else if (strcmp(argv[1],"checkpl") == 0)
	{
		main_checkpl();
	}

	else if (strcmp(argv[1],"checkad") == 0)
	{
		main_checkad();
	}

	else if (strcmp(argv[1],"hfilter") == 0)
	{
		main_hard_filter();
	}
	
	else if (strcmp(argv[1],"filter") == 0)
	{
		main_filter();
	}

	else if (strcmp(argv[1],"fasta") == 0)
	{
		main_fasta();
	}

	else if (strcmp(argv[1],"transcript") == 0)
	{
		main_transcript();
	}

	else if (strcmp(argv[1],"haploview") == 0)
	{
		main_haploview();
	}

	else if (strcmp(argv[1],"arlequin") == 0)
	{
		main_arlequin();
	}
	else if (strcmp(argv[1],"genepop") == 0)
	{
		main_genepop();
	}
	else
    {
        cout << endl << "   >>> WARNING: Command vcfx [" << argv[1] << "] unknown!" << endl;
		main_help();
    }
	
	PrintWarnings();
	auto clock_end = std::chrono::steady_clock::now();
	auto diff = clock_end - clock_start;
	screen_message (screen_size, 2, "Elapsed time: " + to_string(((std::chrono::duration <double, std::milli> (diff).count())/1000)) + " s", 1, v_quiet);
	screen_message (screen_size, 2, "", 1, v_quiet);
    return 0;
}



